<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VirtualCard;
use App\Models\User;
use App\Models\Settings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class VirtualCardController extends Controller
{
    public function index()
    {
        $cards = VirtualCard::with('user')->paginate(20);
        return view('admin.virtual-cards.index', compact('cards'))->with([
            'title' => 'Virtual Cards Management',
            'settings' => Settings::first()
        ]);
    }

    public function create()
    {
        $users = User::all();
        return view('admin.virtual-cards.create', compact('users'))->with([
            'title' => 'Create Virtual Card',
            'settings' => Settings::first()
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'balance' => 'required|numeric|min:0',
            'daily_limit' => 'required|numeric|min:0',
            'monthly_limit' => 'required|numeric|min:0',
        ]);

        VirtualCard::create([
            'user_id' => $request->user_id,
            'card_number' => $this->generateCardNumber(),
            'card_holder_name' => strtoupper(User::find($request->user_id)->name),
            'expiry_month' => '12',
            'expiry_year' => date('Y') + 3,
            'cvv' => rand(100, 999),
            'balance' => $request->balance,
            'daily_limit' => $request->daily_limit,
            'monthly_limit' => $request->monthly_limit,
            'card_type' => 'visa'
        ]);

        return redirect()->route('admin.virtual-cards.index')->with('success', 'Virtual card created successfully!');
    }

    public function updateStatus(Request $request, $id)
    {
        VirtualCard::findOrFail($id)->update(['status' => $request->status]);
        return redirect()->back()->with('success', 'Card status updated successfully!');
    }

    public function updateBalance(Request $request, $id)
    {
        $request->validate(['balance' => 'required|numeric|min:0']);
        VirtualCard::findOrFail($id)->update(['balance' => $request->balance]);
        return redirect()->back()->with('success', 'Card balance updated successfully!');
    }

    public function destroy($id)
    {
        VirtualCard::findOrFail($id)->delete();
        return redirect()->back()->with('success', 'Virtual card deleted successfully!');
    }

    private function generateCardNumber()
    {
        return '4' . str_pad(rand(0, 999999999999999), 15, '0', STR_PAD_LEFT);
    }
}